# telepathy-butterfly - an MSN connection manager for Telepathy
#
# Copyright (C) 2006-2007 Ali Sabil <ali.sabil@gmail.com>
# Copyright (C) 2007 Johann Prieur <johann.prieur@gmail.com>
# Copyright (C) 2009-2010 Collabora, Ltd.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import gobject
import logging
import weakref
import time
import re

import dbus
import telepathy
from telepathy._generated.Channel_Interface_Messages import ChannelInterfaceMessages
from telepathy.interfaces import CHANNEL_INTERFACE_MESSAGES

from tpufa.channel import UfAChannel

__all__ = ['UfATextChannel']

logger = logging.getLogger('UfA.TextChannel')

class UfATextChannel(
        UfAChannel,
        telepathy.server.ChannelTypeText,
        ChannelInterfaceMessages):

    def __init__(self, conn, manager, conversation, handle, props, object_path=None):
        self._handle_name = handle.name
        self._conn_ref = weakref.ref(conn)
        self._conn = conn

        self._pending_messages2 = {}

        telepathy.server.ChannelTypeText.__init__(self, conn, manager, props,
            object_path=object_path)
        UfAChannel.__init__(self, conn, props)
        ChannelInterfaceMessages.__init__(self)

        self._implement_property_get(CHANNEL_INTERFACE_MESSAGES, {
            'SupportedContentTypes': lambda: ["text/plain"] ,
            'MessagePartSupportFlags': lambda: 0,
            'DeliveryReportingSupport': lambda: telepathy.DELIVERY_REPORTING_SUPPORT_FLAG_RECEIVE_FAILURES,
            'PendingMessages': lambda: dbus.Array(self._pending_messages2.values(), signature='aa{sv}')
            })

        self._add_immutables({
            'SupportedContentTypes': CHANNEL_INTERFACE_MESSAGES,
            'MessagePartSupportFlags': CHANNEL_INTERFACE_MESSAGES,
            'DeliveryReportingSupport': CHANNEL_INTERFACE_MESSAGES,
            })

    def _send_text_message(self, message_type, text):
        "Send a simple text message, return true if sent correctly"
        if message_type == telepathy.CHANNEL_TEXT_MESSAGE_TYPE_NORMAL:
            logger.info("Sending message : %s" % unicode(text))
            try:
                self._conn.sendMessage(self._handle_name, text)
            except:
                return False
            return True
        else:
            raise telepathy.NotImplemented("Unhandled message type")
        return False

    def _signal_text_sent(self, timestamp, message_type, text):
        headers = {'message-sent' : timestamp,
                   'message-type' : message_type
                  }
        body = {'content-type': 'text/plain',
                'content': text
               }
        message = [headers, body]
        self.Sent(timestamp, message_type, text)
        self.MessageSent(message, 0, '')

    def _signal_text_received(self, sender, text, id, timestamp, scrollback):
        type = telepathy.CHANNEL_TEXT_MESSAGE_TYPE_NORMAL
        self.Received(id, timestamp, sender.id, type, 0, text)
        headers = dbus.Dictionary({dbus.String('message-received') : dbus.UInt64(timestamp),
                   dbus.String('pending-message-id') : dbus.UInt32(id),
                   dbus.String('message-token') : dbus.String(id),
                   dbus.String('message-sender') : dbus.UInt32(sender),
                   dbus.String('message-type') : dbus.UInt32(type),
                   dbus.String('scrollback') : dbus.Boolean(scrollback)
                  }, signature='sv')

        body = dbus.Dictionary({dbus.String('content-type'): dbus.String('text/plain'),
                dbus.String('content'): dbus.String(text)
               }, signature='sv')
        message = dbus.Array([headers, body], signature='a{sv}')
        self.MessageReceived(message)

    @dbus.service.method(telepathy.CHANNEL_TYPE_TEXT, in_signature='us', out_signature='',
                         async_callbacks=('_success', '_error'))
    def Send(self, message_type, text, _success, _error):
        # The function MUST return before emitting the signals
        _success()
        timestamp = int(time.time())
        self._signal_text_sent(timestamp, message_type, text)

    def Close(self):
        telepathy.server.ChannelTypeText.Close(self)

    def GetPendingMessageContent(self, message_id, parts):
        # We don't support pending message
        raise telepathy.InvalidArgument()

    @dbus.service.method(telepathy.CHANNEL_INTERFACE_MESSAGES, in_signature='aa{sv}u',
                         out_signature='s', async_callbacks=('_success', '_error'))
    def SendMessage(self, message, flags, _success, _error):
        headers = message.pop(0)
        message_type = int(headers.get('message-type', telepathy.CHANNEL_TEXT_MESSAGE_TYPE_NORMAL))
        if message_type != telepathy.CHANNEL_TEXT_MESSAGE_TYPE_NORMAL:
                raise telepathy.NotImplemented("Unhandled message type")
        text = None
        for part in message:
            if part.get("content-type", None) ==  "text/plain":
                text = part['content']
                break
        if text is None:
                raise telepathy.NotImplemented("Unhandled message type")

        if self._send_text_message(message_type, text):
            timestamp = int(time.time())
            # The function MUST return before emitting the signals
            _success('')
            self._signal_text_sent(timestamp, message_type, text)

    def AcknowledgePendingMessages(self, ids):
        readIds = [] 
        for id in ids:
            if id in self._pending_messages2:
                del self._pending_messages2[id]
                readIds.append(str(id))
        if readIds != []:
            self._conn.markSMSAsRead(readIds)

        telepathy.server.ChannelTypeText.AcknowledgePendingMessages(self, ids)
        self.PendingMessagesRemoved(ids)

    def ListPendingMessages(self, clear):
        if clear:
            ids = self._pending_messages2.keys()
            self._pending_messages2 = {}
            self.PendingMessagesRemoved(ids)

        return telepathy.server.ChannelTypeText.ListPendingMessages(self, clear)

    def GetSelfHandle(self):
        return self._conn.GetSelfHandle()

    @dbus.service.signal(telepathy.CHANNEL_INTERFACE_MESSAGES, signature='aa{sv}')
    def MessageReceived(self, message):
        id = message[0]['pending-message-id']
        self._pending_messages2[id] = dbus.Array(message, signature='a{sv}')
