# telepathy-ufa - A Ubuntu For Android connection manager
#
# Copyright (C) 2012 Canonical Ltd.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import logging
import weakref
import dbus

import telepathy

from tpufa.channel import UfAChannel
from tpufa.android.androidcall import AndroidCall
from tpufa.call import UfACallContent, UfAPhone, UfACall, UfACallConnection

from telepathy.interfaces import CHANNEL_INTERFACE, CHANNEL_TYPE_CALL, CALL_INTERFACE_MUTE, CHANNEL_INTERFACE_HOLD
from telepathy.constants import LOCAL_MUTE_STATE_UNMUTED, LOCAL_MUTE_STATE_MUTED, LOCAL_HOLD_STATE_REASON_REQUESTED, LOCAL_HOLD_STATE_UNHELD, LOCAL_HOLD_STATE_HELD
from telepathy._generated.Call_Interface_Mute import CallInterfaceMute
from telepathy._generated.Channel_Interface_Hold import ChannelInterfaceHold

__all__ = ['UfACallChannel']

logger = logging.getLogger('UfA.CallChannel')

# FIXME: we need to implement also Call1.Interface.Mute
class UfACallChannel(
        UfAChannel,
        telepathy.server.ChannelTypeCall,
        CallInterfaceMute,
        ChannelInterfaceHold,
        AndroidCall):

    __call_content_id = 1
    phone = None

    def __init__(self, conn, manager, call, handle, props, object_path=None):
        telepathy.server.ChannelTypeCall.__init__(self, conn, manager, props,
            object_path=object_path)
        UfAChannel.__init__(self, conn, props)
        AndroidCall.__init__(self)
        ChannelInterfaceHold.__init__(self)

        self.phone = UfAPhone(self.getDefaultPhone())
        self._conn = conn
        self._channel_manager = manager
        self._call = call
        self._handle = handle
        self._memberFlags = 0
        self._waitDialing = False
        self._androidCallId = None
        self._pendingHangup = False
        self._pendingSpeakerMode = None
        self._localMuteState = LOCAL_MUTE_STATE_UNMUTED
        self._holdState = LOCAL_HOLD_STATE_UNHELD 
        self._androidCallId = self._handle.name
        self._requested = props.get(telepathy.CHANNEL_INTERFACE + ".Requested", True)
        self._initiatorHandle = props.get(telepathy.CHANNEL_INTERFACE + '.InitiatorHandle', None)

        if not self._initiatorHandle or self._requested:
            self._initiatorHandle = self._conn.self_handle.id

        self._incoming = (self._initiatorHandle != self._conn.self_handle.id)

        self._initialAudio = props.get(CHANNEL_TYPE_CALL + '.InitialAudio', True)
        self._initialVideo = props.get(CHANNEL_TYPE_CALL + '.InitialVideo', False)
        self._content = []
        self._speakerMode = self.isSpeakerOn()

        self._add_immutables({
            'HardwareStreaming': CHANNEL_TYPE_CALL,
            'InitialAudio': CHANNEL_TYPE_CALL,
            'InitialVideo': CHANNEL_TYPE_CALL,
        })

        self._implement_property_get(CHANNEL_TYPE_CALL, {
            'Contents': lambda: dbus.Array(self._content, signature="o"),
            'CallStateDetails': lambda: self.GetCallStateDetails(),
            'CallState': lambda: dbus.UInt32(self._state),
            'CallFlags': lambda: dbus.UInt32(self._flags),
            'CallStateReason': lambda: self.GetCallStateReason(),
            'HardwareStreaming': lambda: dbus.Boolean(True),
            'CallMembers': lambda: self.GetCallMembers(),
            'MemberIdentifiers': lambda: self.GetMemberIdentifiers(),
            'InitialTransport': lambda: dbus.UInt32(telepathy.STREAM_TRANSPORT_TYPE_UNKNOWN),
            'InitialAudio': lambda: dbus.Boolean(self._initialAudio),
            'InitialVideo': lambda: dbus.Boolean(self._initialVideo),
            'SpeakerMode': lambda: dbus.Boolean(self._speakerMode),
        })

        self._implement_property_get(CALL_INTERFACE_MUTE, {
            'LocalMuteState': lambda: dbus.UInt32(self._localMuteState),
        })

        self._implement_property_get(CHANNEL_INTERFACE_HOLD, {
            'GetHoldState': self.GetHoldState()
        })

        self.setCallState(telepathy.CALL_STATE_INITIALISING, 0, telepathy.CALL_STATE_CHANGE_REASON_PROGRESS_MADE, 0)

        # Add content if any
        if self._initialAudio:
            self.AddContent("audio", telepathy.MEDIA_STREAM_TYPE_AUDIO, telepathy.MEDIA_STREAM_DIRECTION_BIDIRECTIONAL)

        if self._initialVideo:
            self.AddContent("video", telepathy.MEDIA_STREAM_TYPE_VIDEO, telepathy.MEDIA_STREAM_DIRECTION_BIDIRECTIONAL)

        # if it is not an incoming call, we need to call android side
        if not self._incoming:
            # make sure we are not already active before calling
            if not self.isAndroidForeground() and not self.isAndroidBackground() and not self.isAndroidRinging() and self._requested:
                logger.info("Going to call %s" % self._handle.name)
                self.createCall(self._handle.name)
            # set waitDialing so then we dont process
            # any signals until we get the right id from android
            if self._requested:
                self._waitDialing = True
            self.setMemberFlags(self._handle.id, telepathy.CALL_MEMBER_FLAG_RINGING, reason=telepathy.CALL_STATE_CHANGE_REASON_PROGRESS_MADE)
        else:
            self.setMemberFlags(self._handle.id, 0, reason=telepathy.CALL_STATE_CHANGE_REASON_PROGRESS_MADE)

        self.setCallState(telepathy.CALL_STATE_INITIALISED, 0, telepathy.CALL_STATE_CHANGE_REASON_PROGRESS_MADE, 0)

    def SetRinging(self):
        logger.info("SetRinging called");
        if self._incoming and self._state == telepathy.CALL_STATE_INITIALISED:
            self.setCallState(self._state, self._conn.self_handle.id, 
                telepathy.CALL_STATE_CHANGE_REASON_USER_REQUESTED, telepathy.CALL_FLAG_LOCALLY_RINGING)
        # FIXME: we need to return the Not Available error for some cases.
        #        check this documentation: http://telepathy.freedesktop.org/spec/Channel_Type_Call.html#Method:SetRinging

    def SetQueued(self):
        logger.info("SetQueued called");
        # TODO: implement

    def Accept(self):
        logger.info("Accept called");
        if self._incoming:
            self.setCallState(telepathy.CALL_STATE_ACCEPTED, self._conn.self_handle.id,
                telepathy.CALL_STATE_CHANGE_REASON_USER_REQUESTED, 0);
            self.acceptCall()

    def Hangup(self, Reason, Detailed_Hangup_Reason, Message):
        logger.info("Hangup called");

        # check if this call is active or ringing
        if self._state == telepathy.CALL_STATE_ACTIVE or self._state == telepathy.CALL_STATE_ACCEPTED or self._state == telepathy.CALL_STATE_INITIALISED:
            if self._incoming and self._state == telepathy.CALL_STATE_INITIALISED:
                logger.info("Rejecting the incoming call")
                self.rejectCall()
                return
            logger.info("Hangup active call")
            if self.isAndroidBackground():
                logger.info("Hanging up a background call")
                self.hangupBackgroundCall()
            elif self.isAndroidForeground():
                logger.info("Hanging up a foreground call")
                self.hangupForegroundCall()
            else:
                # this call probably wasn't registered on android yet
                self._pendingHangup = True

    def AddContent(self, Content_Name, Content_Type, InitialDirection):
        logger.info("AddContent called");
        path = "%s/content%d" % (self._object_path, self.__call_content_id)
        content = UfACallContent(self, Content_Name, Content_Type, self._conn._name, object_path=path)
        self.__call_content_id += 1
        self.ContentAdded(content)
        self._content.append(content)

    def GetCallStateDetails(self):
        details = dbus.Dictionary({}, signature="sv")
        logger.info("GetCallStateDetails called");
        return details

    def GetCallStateReason(self):
        logger.info("GetCallStateReason called");
        return (dbus.UInt32(self._stateActor), dbus.UInt32(self._stateReason), dbus.String(""), dbus.String(""))

    def GetCallMembers(self):
        members = dbus.Dictionary({}, signature="uu")
        logger.info("GetCallMembers called");
        members[dbus.UInt32(self._handle.id)] = dbus.UInt32(self._memberFlags)
        return members

    def GetMemberIdentifiers(self):
        identifiers = dbus.Dictionary({}, signature="us")
        identifiers[dbus.UInt32(self._handle.id)] = self._handle.name
        logger.info("GetMemberIdentifiers called");
        return identifiers

    def Close(self):
        self.Closed()

        for content in self._content:
            content.Remove()

        try:
            self._channel_manager.remove_channel(self)
            self._conn.remove_channel(self)
            # stop listening to signals so the
            # garbage collector can delete this instance
            self.unregister_from_dbus()
            self.remove_from_connection()
        except:
            pass

    def removeContent(self, content):
        self.ContentRemoved(content, (dbus.UInt32(self._conn.self_handle.id), dbus.UInt32(telepathy.CALL_STATE_CHANGE_REASON_USER_REQUESTED), dbus.String(""), dbus.String(""))) 
        del self._content[self._content.index(content)]

    def setCallState(self, state, actor, reason, flags):
        self._state = state
        self._stateActor = actor
        self._stateReason = reason
        self._flags = flags

        # signature='uu(uuss)a{sv}')
        self.CallStateChanged(dbus.UInt32(self._state), dbus.UInt32(self._flags), 
            (dbus.UInt32(self._stateActor), dbus.UInt32(self._stateReason), dbus.String(""), dbus.String("")), 
            dbus.Dictionary({}, signature="sv"))

    def setMemberFlags(self, member, flags, actor=0, reason=telepathy.CALL_STATE_CHANGE_REASON_UNKNOWN):
        # FIXME: handle multiple contacts
        self._memberFlags = flags

        # signature='a{uu}a{us}au(uuss)')
        self.CallMembersChanged(dbus.Dictionary({dbus.UInt32(self._handle.id) : dbus.UInt32(self._memberFlags)}, signature="uu"),
            dbus.Dictionary({dbus.UInt32(self._handle.id) : dbus.String(self._handle.name)}, signature="us"),
            dbus.Array([], signature="u"),
            (dbus.UInt32(actor), dbus.UInt32(reason), dbus.String(""), dbus.String("")))

    def isAndroidRinging(self):
        # check how many connections (should be max 1)
        if (len(self.phone.ringingCall.connections) == 1):
            conn = self.phone.ringingCall.connections[0]
            if conn.id == self._androidCallId:
                return True
        return False

    def isAndroidBackground(self):
        if (len(self.phone.backgroundCall.connections) != 0):
            for conn in self.phone.backgroundCall.connections:
                if conn.id == self._androidCallId:
                    return True
        return False

    def isAndroidForeground(self):
        if (len(self.phone.foregroundCall.connections) != 0):
            for conn in self.phone.foregroundCall.connections:
                if conn.id == self._androidCallId:
                    return True
        return False

    def isAndroidDialing(self):
        if (len(self.phone.foregroundCall.connections) != 0):
            if self.phone.foregroundCall.state == self.CALL_STATE_DIALING:
                return True
        return False

    def isAndroidActive(self):
        if (len(self.phone.foregroundCall.connections) != 0):
            if self.phone.foregroundCall.state == self.CALL_STATE_ACTIVE:
                return True
        return False

    ### Android Call Interface ###
    def AndroidCallStateChanged(self, calls):
        logger.info("Incoming call: %s", self._incoming)

        # update the phone struct
        self.phone = UfAPhone(calls)

        # hack to get the right id when dialing a contact
        if (self._waitDialing and self.isAndroidDialing()):
            self._androidCallId = self.phone.foregroundCall.connections[0].id
            self._waitDialing = False
        elif (self._waitDialing and self.isAndroidActive()):
            # when using apk, it goes to active directly
            self._waitDialing = False
            self._androidCallId = self.phone.foregroundCall.connections[0].id
 
        # wait until we get the right id
        if self._waitDialing:
            return

        if self._pendingSpeakerMode != None:
            self.turnOnSpeaker(self._pendingSpeakerMode)

        if self._pendingHangup:
            logger.info("Execute pending hangup")
            self._pendingHangup = False
            self.Hangup(telepathy.CALL_STATE_CHANGE_REASON_USER_REQUESTED, None, None)
            return

        if self._incoming:
            if self.isAndroidRinging():
                # still ringing
                logger.info("The call is still ringing")
                return
            elif self.isAndroidBackground():
                # put on hold
                logger.info("The call is on hold")
                if (self._holdState == LOCAL_HOLD_STATE_UNHELD):
                    self.HoldStateChanged(LOCAL_HOLD_STATE_HELD, LOCAL_HOLD_STATE_REASON_REQUESTED)
                    self._holdState = LOCAL_HOLD_STATE_HELD
                    self.setCallState(telepathy.CALL_STATE_ACTIVE, self._handle.id, telepathy.CALL_STATE_CHANGE_REASON_USER_REQUESTED, telepathy.CALL_FLAG_LOCALLY_HELD) 
            elif self.isAndroidForeground():
                # the call was accepted/unheld
                logger.info("The call is active")
                if (self._holdState == LOCAL_HOLD_STATE_HELD):
                    self.HoldStateChanged(LOCAL_HOLD_STATE_UNHELD, LOCAL_HOLD_STATE_REASON_REQUESTED)
                    self._holdState = LOCAL_HOLD_STATE_UNHELD
                self.setCallState(telepathy.CALL_STATE_ACTIVE, self._handle.id, telepathy.CALL_STATE_CHANGE_REASON_USER_REQUESTED, 0)
            else:
                logger.info("The call is ended")
                if self._state == telepathy.CALL_STATE_ACTIVE:
                    self.setCallState(telepathy.CALL_STATE_ENDED, self._handle.id, telepathy.CALL_STATE_CHANGE_REASON_USER_REQUESTED, 0)
                else:
                    # incoming call stopped ringing
                    self.setCallState(telepathy.CALL_STATE_ENDED, self._handle.id, telepathy.CALL_STATE_CHANGE_REASON_NO_ANSWER, 0)
                self.Close()
        else: # handle outgoing call
            if self.isAndroidBackground():
                # put on hold
                logger.info("The call is on hold")
                if (self._holdState == LOCAL_HOLD_STATE_UNHELD):
                    self.HoldStateChanged(LOCAL_HOLD_STATE_HELD, LOCAL_HOLD_STATE_REASON_REQUESTED)
                    self._holdState = LOCAL_HOLD_STATE_HELD
                    self.setCallState(telepathy.CALL_STATE_ACTIVE, self._handle.id, telepathy.CALL_STATE_CHANGE_REASON_USER_REQUESTED, telepathy.CALL_FLAG_LOCALLY_HELD) 
            elif self.isAndroidForeground():
                if self._state == telepathy.CALL_STATE_ACCEPTED:
                    logger.info("The call is waiting for answer")
                else:
                    # check if this connection became active
                    status = self.phone.foregroundCall.state
                    if status == self.CALL_STATE_ACTIVE:
                        logger.info("The call is active")
                        if (self._holdState == LOCAL_HOLD_STATE_HELD):
                            self.HoldStateChanged(LOCAL_HOLD_STATE_UNHELD, LOCAL_HOLD_STATE_REASON_REQUESTED)
                            self._holdState = LOCAL_HOLD_STATE_UNHELD
                        self.setCallState(telepathy.CALL_STATE_ACTIVE, self._handle.id, telepathy.CALL_STATE_CHANGE_REASON_USER_REQUESTED, 0)
            elif self.isAndroidRinging():
                logger.info("The call is ringing")
                return
            else:
                logger.info("The call is ended")
                self.setCallState(telepathy.CALL_STATE_ENDED, self._handle.id, telepathy.CALL_STATE_CHANGE_REASON_NO_ANSWER, 0)
                self.Close()

    def RequestMuted(self, muted):
        logger.info("RequestMuted called")
        if (self._localMuteState != LOCAL_MUTE_STATE_MUTED and muted == True):
            self._localMuteState = LOCAL_MUTE_STATE_MUTED
            self.MuteStateChanged(self._localMuteState)
            self.setMute(True)
        elif (self._localMuteState != LOCAL_MUTE_STATE_UNMUTED and muted == False):
            self._localMuteState = LOCAL_MUTE_STATE_UNMUTED
            self.MuteStateChanged(self._localMuteState)
            self.setMute(False)

    def RequestHold(self, hold):
        logger.info("RequestHold called")
        if (self._holdState != LOCAL_HOLD_STATE_HELD and hold == True):
            self.switchHoldingAndActive()
        elif (self._holdState != LOCAL_HOLD_STATE_UNHELD and hold == False):
            self.switchHoldingAndActive()

    def GetHoldState(self):
        return dbus.UInt32(self._holdState), dbus.UInt32(LOCAL_HOLD_STATE_REASON_REQUESTED)

    def MuteSetChanged(self):
        # ignore signal if this call is not in foreground
        if not self.isAndroidForeground():
            return
        if (self.getMute()):
            self._localMuteState = LOCAL_MUTE_STATE_MUTED
            self.MuteStateChanged(self._localMuteState)
        else:
            self._localMuteState = LOCAL_MUTE_STATE_UNMUTED
            self.MuteStateChanged(self._localMuteState)

    def SpeakerSetChanged(self):
        self._speakerMode = self.isSpeakerOn()
        self.SpeakerChanged(self._speakerMode)

    @dbus.service.signal(dbus_interface=CHANNEL_TYPE_CALL,
                         signature='b')
    def SpeakerChanged(self, state):
        pass

    @dbus.service.method(dbus_interface=CHANNEL_TYPE_CALL, in_signature='b')
    def turnOnSpeaker(self, active):
        # postpone speaker mode until the call is displayed on android
        if self._state == telepathy.CALL_STATE_INITIALISED and self._pendingSpeakerMode == None:
            self._pendingSpeakerMode = active
        else:
            self._pendingSpeakerMode = None
            AndroidCall.turnOnSpeaker(self, active, False)

