# telepathy-ufa - A Ubuntu For Android connection manager
#
# Copyright (C) 2012 Canonical Ltd.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import logging
import weakref
import dbus

import telepathy
from telepathy.interfaces import CALL_CONTENT, CALL_CONTENT_INTERFACE_DTMF
from telepathy._generated.Call_Content import CallContent
from telepathy._generated.Call_Content_Interface_DTMF import CallContentInterfaceDTMF

__all__ = ['UfACallContent.py']

logger = logging.getLogger('UfA.CallContent')

class UfACallContent(CallContent, 
                     telepathy.server.DBusProperties, 
                     CallContentInterfaceDTMF):

    def __init__(self, channel, name, contentType, conn, object_path=None, bus_name=None):
        CallContent.__init__(self, conn, object_path=object_path, bus_name=bus_name)
        telepathy.server.DBusProperties.__init__(self)
        CallContentInterfaceDTMF.__init__(self)

        self._channel = channel
        self._type = contentType
        self._name = name
        self._disposition = telepathy.CALL_CONTENT_DISPOSITION_NONE

        self._add_immutable_properties({
            'Name': CALL_CONTENT,
            'Type': CALL_CONTENT,
            'Disposition': CALL_CONTENT,
            'Interfaces' : CALL_CONTENT,
        })

        self._implement_property_get(CALL_CONTENT, {
            'Name': lambda: dbus.String(self._name),
            'Type': lambda: dbus.UInt32(self._type),
            'Disposition': lambda: dbus.UInt32(self._disposition),
            'Streams': lambda: dbus.Array([], signature="o"),
            'Interfaces' : lambda: dbus.Array(self._interfaces, signature="s"),
        })

    def Remove(self):
        self._channel.removeContent(self)
        self.remove_from_connection()
        self._channel = None 

    def StartTone(self, event):
        logger.info("StartTone called")
        androidEvent = event
        if event == 10:
            # 42 is '#' in ascii table
            androidEvent = 42
        elif event == 11:
            # 35 is '*' in ascii table
            androidEvent = 35
        else:
            # convert number to ascii value
            androidEvent = event + 48
        
        self._channel.sendDtmf(androidEvent)

    def StopTone(self):
        logger.info("StopTone called")
        pass

