/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <netbook-launcher/netbook-launcher.h>
#include "nl-plugin-manager.h"

#include <glib.h>
#include <glib/gi18n.h>

G_DEFINE_TYPE (NlPluginManager, nl_plugin_manager, G_TYPE_OBJECT);

#define NL_PLUGIN_MANAGER_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  NL_TYPE_PLUGIN_MANAGER, \
  NlPluginManagerPrivate))

struct _NlPluginManagerPrivate
{
  GSList        *plugins;
  GSList        *modules;
  NlShell *shell;
};

enum
{
  PROP_0,
  PROP_SHELL
};

/* Forwards */

/* GObject stuff */
static void
set_property (GObject      *object,
              guint         prop_id,
              const GValue *value,
              GParamSpec   *pspec)
{
  NlPluginManagerPrivate *priv;

  g_return_if_fail (NL_IS_PLUGIN_MANAGER (object));
  priv = NL_PLUGIN_MANAGER_GET_PRIVATE (object);

  switch (prop_id)
    {
    case PROP_SHELL:
      priv->shell = g_value_get_pointer (value);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
get_property (GObject      *object,
              guint         prop_id,
              GValue       *value,
              GParamSpec   *pspec)
{
  NlPluginManagerPrivate *priv;

  g_return_if_fail (NL_IS_PLUGIN_MANAGER (object));
  priv = NL_PLUGIN_MANAGER_GET_PRIVATE (object);

  switch (prop_id)
    {
    case PROP_SHELL:
      g_value_set_pointer (value, priv->shell);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
nl_plugin_manager_finalize (GObject *object)
{
  NlPluginManagerPrivate *priv;

  priv = NL_PLUGIN_MANAGER_GET_PRIVATE (object);

  if (priv->plugins)
    {
      g_slist_foreach (priv->plugins, (GFunc)g_object_unref, NULL);
      g_slist_free (priv->plugins);
      priv->plugins = NULL;
    }

  if (priv->modules)
    {
      g_slist_foreach (priv->modules, (GFunc)g_module_close, NULL);
      g_slist_free (priv->modules);
      priv->modules = NULL;
    }

  G_OBJECT_CLASS (nl_plugin_manager_parent_class)->finalize (object);
}

static void
nl_plugin_manager_constructed (GObject *object)
{
  NlPluginManagerPrivate *priv;
  GDir        *dir;
  const gchar *name;
  GError      *error = NULL;

  priv = NL_PLUGIN_MANAGER_GET_PRIVATE (object);

  dir = g_dir_open (PLUGINDIR, 0, &error);

  if (error)
    {
      g_debug ("Not loading modules: %s", error->message);
      g_error_free (error);
      return;
    }

  while ((name = g_dir_read_name (dir)) != NULL)
    {
      GModule *module;
      gchar   *fullpath;
      nl_plugin_new_t new_func;

      if (!g_str_has_suffix (name, G_MODULE_SUFFIX))
        continue;

      g_debug ("Loading Module: %s\n", name);

      fullpath = g_build_filename (PLUGINDIR, name, NULL);
      module = g_module_open (fullpath,
                              G_MODULE_BIND_LAZY | G_MODULE_BIND_LOCAL);

      if (!module)
        {
          g_warning ("Unable to load module: %s", fullpath);
          g_free (fullpath);
          continue;
        }

      if (g_module_symbol (module, NL_PLUGIN_INIT_SYMBOL,
                           (gpointer)&new_func))
        {
          GObject *plugin = new_func (priv->shell);

          if (!G_IS_OBJECT (plugin))
            {
              g_warning ("Unable to initialize module %s", name);
              g_module_close (module);
              g_free (fullpath);
              continue;
            }

          g_module_make_resident (module);

          priv->plugins = g_slist_append (priv->plugins, plugin);
          priv->modules = g_slist_append (priv->modules, module);
          g_free (fullpath);
        }
      else
        {
          g_warning ("Unable to find symbol %s in module %s",
                     NL_PLUGIN_INIT_SYMBOL, name);
          g_free (fullpath);
          g_module_close (module);
          continue;
        }
    }

  g_dir_close (dir);
}

static void
nl_plugin_manager_class_init (NlPluginManagerClass *klass)
{
  GObjectClass *obj_class = G_OBJECT_CLASS (klass);
  GParamSpec   *pspec;

  obj_class->finalize     = nl_plugin_manager_finalize;
  obj_class->set_property = set_property;
  obj_class->get_property = get_property;
  obj_class->constructed  = nl_plugin_manager_constructed;

  pspec = g_param_spec_pointer ("shell", "shell", "shell",
                                G_PARAM_READWRITE | G_PARAM_CONSTRUCT);
  g_object_class_install_property (obj_class, PROP_SHELL, pspec);

  g_type_class_add_private (obj_class, sizeof (NlPluginManagerPrivate));
}

static void
nl_plugin_manager_init (NlPluginManager *manager)
{
  NlPluginManagerPrivate *priv;

  priv = manager->priv = NL_PLUGIN_MANAGER_GET_PRIVATE (manager);

  priv->plugins = NULL;
  priv->modules = NULL;
  priv->shell = NULL;
}

/*
 * Public methods
 */
NlPluginManager *
nl_plugin_manager_new (NlShell *s)
{
  NlPluginManager *plugin_manager = NULL;

  g_return_val_if_fail (NL_IS_SHELL (s), NULL);

  if (!plugin_manager)
    plugin_manager = g_object_new (NL_TYPE_PLUGIN_MANAGER,
                                   "shell", s,
                                   NULL);

  return plugin_manager;
}
