# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Olivier Tilloy <olivier@fluendo.com>

"""
Common models related to media.
"""

from elisa.core.components.model import Model


class PlayableModel(Model):

    """
    A playable model contains the real URI of a playable media, that is,
    typically, a URI directly usable by gstreamer to play streamed data.
    It also potentially contains a title for the media.

    @ivar uri:   the real URI of the playable media
    @type uri:   L{elisa.core.media_uri.MediaUri}
    @ivar title: the title of the playable media
    @type title: C{unicode}
    """

    def __init__(self):
        """
        Constructor. Initialize all the fields.
        """
        super(PlayableModel, self).__init__()
        self.uri = None
        self.title = None

    def __eq__(self, other):
        """
        Compare two playable models.

        Two playable models are considered equal if and only if their URIs are
        equal.

        @param other: another playable model
        @type other:  L{elisa.plugins.base.models.media.PlayableModel}

        @return:      C{True} if the two models are equal, C{False} otherwise
        @rtype:       C{bool}
        """
        return (self.uri == other.uri)


class RawDataModel(Model):

    """
    A raw data model contains raw data from a media file (can be binary, text,
    etc...).

    @ivar data: the raw data
    @type data: C{str}
    @ivar size: the total size of the data
    @type size: C{int}
    """

    def __init__(self):
        """
        Constructor. Initialize all the fields.
        """
        super(RawDataModel, self).__init__()
        self.data = None
        self.size = None


# TODO: subclass the RawDataModel to have a model that reads its data
#       asynchronously.
