#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <string.h>

#include "E_Nm.h"
#include "e_nm_private.h"
#include "E_DBus.h"
#include "e_dbus_private.h"

#define CHECK_SIGNATURE(msg, err, sig)                       \
  if (dbus_error_is_set((err)))                              \
  {                                                          \
    ERR("%s - %s", (err)->name, (err)->message);  \
    return NULL;                                             \
  }                                                          \
                                                             \
  if (!dbus_message_has_signature((msg), (sig)))             \
  {                                                          \
    dbus_set_error((err), DBUS_ERROR_INVALID_SIGNATURE, ""); \
    return NULL;                                             \
  }

static E_NM_Variant *property_string(DBusMessageIter *iter, const char *sig, void *value);
static E_NM_Variant *property_basic(DBusMessageIter *iter, const char *sig, void *value);
static E_NM_Variant *property_variant(DBusMessageIter *iter, const char *sig, void *value);
static E_NM_Variant *property_array(DBusMessageIter *iter, const char *sig, void *value);
static void          property_free(E_NM_Variant *var);

typedef E_NM_Variant *(*Property_Cb)(DBusMessageIter *iter, const char *sig, void *value);

typedef struct Sig_Property Sig_Property;
struct Sig_Property
{
  const char *sig;
  Property_Cb func;
};

static const Sig_Property sigs[] = {
  { .sig = "s", property_string },
  { .sig = "o", property_string },
  { .sig = "u", property_basic },
  { .sig = "b", property_basic },
  { .sig = "y", property_basic },
  { .sig = "t", property_basic },
  { .sig = "v", property_variant },
  { .sig = "a", property_array },
  { .sig = "as", property_array },
  { .sig = "ao", property_array },
  { .sig = "ay", property_array },
  { .sig = "au", property_array },
  { .sig = "aau", property_array },
  { .sig = NULL }
};

static const Property_Cb
find_property_cb(const char *sig)
{
  const Sig_Property *t;

  if (!sig) return NULL;

  for (t = sigs; t->sig; t++)
  {
    if (!strcmp(t->sig, sig))
      return t->func;
  }
  ERR("Missing property parser for sig: %s", sig);
  return NULL;
}

static const Property *
find_property(const char *name, const Property *properties)
{
  const Property *p;

  if (!name) return NULL;

  for (p = properties; p->name; p++)
  {
    if (!strcmp(p->name, name))
      return p;
  }
  return NULL;
}

static E_NM_Variant *
property_string(DBusMessageIter *iter, const char *sig, void *value)
{
  const char   *str;
  E_NM_Variant *var = NULL;

  if ((value) && (!sig))
  {
    ERR("Can't have value and no sig");
    return NULL;
  }
  dbus_message_iter_get_basic(iter, &str);
  if (sig)
  {
    if (!check_arg_type(iter, sig[0])) return NULL;
    if (!value) value = &var;
  }
  else
  {
    var = malloc(sizeof(E_NM_Variant));
    var->type = dbus_message_iter_get_arg_type(iter);
    value = &var->s;
  }
  *((char **)value) = strdup(str);

  return var;
}

static E_NM_Variant *
property_basic(DBusMessageIter *iter, const char *sig, void *value)
{
  E_NM_Variant *var = NULL;

  if ((value) && (!sig))
  {
    ERR("Can't have value and no sig");
    return NULL;
  }
  if (sig)
  {
    if (!check_arg_type(iter, sig[0])) return NULL;
    if (!value)
    {
      // TODO: Only alloc right size
      var = malloc(sizeof(long long));
      value = var;
    }
  }
  else
  {
    var = malloc(sizeof(E_NM_Variant));
    var->type = dbus_message_iter_get_arg_type(iter);
    value = var;
  }
  dbus_message_iter_get_basic(iter, value);

  return var;
}

static E_NM_Variant *
property_variant(DBusMessageIter *iter, const char *sig __UNUSED__, void *value __UNUSED__)
{
  DBusMessageIter v_iter;
  Property_Cb     func;
  char            tmp[2];

  if (!check_arg_type(iter, 'v')) return NULL;
  dbus_message_iter_recurse(iter, &v_iter);
  tmp[0] = dbus_message_iter_get_arg_type(&v_iter);
  tmp[1] = 0;
  func = find_property_cb(tmp);
  if (!func) return NULL;
  return (*func)(&v_iter, NULL, NULL);
}

static E_NM_Variant *
property_array(DBusMessageIter *iter, const char *sig, void *value)
{
  DBusMessageIter   a_iter;
  Eina_List      **list;
  Property_Cb       func;
  E_NM_Variant    *var = NULL;
  const char       *subsig = NULL;

  if ((value) && (!sig))
  {
    ERR("Can't have value and no sig");
    return NULL;
  }

  dbus_message_iter_recurse(iter, &a_iter);
  if (sig)
  {
    if (!check_arg_type(iter, sig[0])) return NULL;
    subsig = (sig + 1);
    func = find_property_cb(subsig);
    if (!func) return NULL;
    if (!value) value = &var;
    list = (Eina_List **)value;
    *list = NULL;
  }
  else
  {
    char tmp[] = { dbus_message_iter_get_arg_type(&a_iter), 0 };
    func = find_property_cb(tmp);
    if (!func) return NULL;
    var = malloc(sizeof(E_NM_Variant));
    var->type = dbus_message_iter_get_arg_type(iter);
    list = (Eina_List **)&var->a;
    *list = NULL;
  }

  while (dbus_message_iter_get_arg_type(&a_iter) != DBUS_TYPE_INVALID)
  {
    void *subvar;

    subvar = (*func)(&a_iter, subsig, NULL);
    if (subvar) *list = eina_list_append(*list, subvar);
    dbus_message_iter_next(&a_iter);
  }

  return var;
}

static void
property_free(E_NM_Variant *var)
{
  if (!var) return;
  if ((var->type == 's') || (var->type == 'o'))
    free(var->s);
  else if (var->type == 'a')
  {
    E_NM_Variant *v;

    EINA_LIST_FREE(var->a, v)
      property_free(v);
  }
  free(var);
}

int
property_get(E_DBus_Connection *conn, Property_Data *data)

{
  return e_dbus_properties_get(conn, data->service, data->object, data->interface, data->property->name, property, data) ? 1 : 0;
}

void
property(void *data, DBusMessage *msg, DBusError *err)
{
  DBusMessageIter  iter, v_iter;
  Property_Data   *d;
  void            *value;
  Property_Cb      func = NULL;

  d = data;
  if (dbus_error_is_set(err))
  {
    ERR("%s - %s", err->name, err->message);
    goto error;
  }
  if (!dbus_message_has_signature(msg, "v")) goto error;
  dbus_message_iter_init(msg, &iter);
  dbus_message_iter_recurse(&iter, &v_iter);
  if (d->property->func)
  {
    d->property->func(d, &v_iter);
    return;
  }

  value = ((char *)d->reply + d->property->offset);
  func = find_property_cb(d->property->sig);
  if (func) (*func)(&v_iter, d->property->sig, value);

  d->property++;
  if (d->property->name)
    e_dbus_properties_get(d->nmi->conn, d->service, d->object, d->interface, d->property->name, property, d);
  else
  {
    if (d->cb_func) d->cb_func(d->data, d->reply);
    property_data_free(d);
  }
  return;

error:
  if (d->reply) free(d->reply); /* TODO: Correct free for object */
  if (d->cb_func) d->cb_func(d->data, NULL);
  property_data_free(d);
}

void
parse_properties(void *data, const Property *properties, DBusMessage *msg)
{
  DBusMessageIter iter, a_iter;

  if (!dbus_message_has_signature(msg, "a{sv}")) return;

  dbus_message_iter_init(msg, &iter);

  dbus_message_iter_recurse(&iter, &a_iter);
  while (dbus_message_iter_get_arg_type(&a_iter) != DBUS_TYPE_INVALID)
  {
    DBusMessageIter d_iter, v_iter;
    const Property *p;
    Property_Cb func;
    const char *name;
    void *value;

    dbus_message_iter_recurse(&a_iter, &d_iter);
    if (!check_arg_type(&d_iter, 's')) return;
    dbus_message_iter_get_basic(&d_iter, &name);

    dbus_message_iter_next(&d_iter);
    if (!check_arg_type(&d_iter, 'v')) return;
    dbus_message_iter_recurse(&d_iter, &v_iter);

    p = find_property(name, properties);
    if (!p) goto next;
    value = ((char *)data + p->offset);
    func = find_property_cb(p->sig);
    if (!func) goto next;
    func(&v_iter, p->sig, value);

next:
    dbus_message_iter_next(&a_iter);
  }
}

#if 0
/**
 * @internal
 * @brief Generic callback for methods that return nothing
 */
void *
cb_nm_generic(DBusMessage *msg, DBusError *err)
{
  return NULL;
}

/**
 * @internal
 * @brief Generic free for methods
 */
void
free_nm_generic(void *data)
{
  if (!data) return;
  free(data);
}

/**
 * @internal
 * @brief Callback for methods that return DBUS_TYPE_INT32
 */
void *
cb_nm_int32(DBusMessage *msg, DBusError *err)
{
  dbus_int32_t *i;

  CHECK_SIGNATURE(msg, err, "i");

  i = malloc(sizeof(dbus_int32_t));
  /* Actually emit the integer */
  dbus_message_get_args(msg, err,
                        DBUS_TYPE_INT32, i,
                        DBUS_TYPE_INVALID);

  return i;
}

/**
 * @internal
 * @brief Callback for methods that return DBUS_TYPE_UINT32
 */
void *
cb_nm_uint32(DBusMessage *msg, DBusError *err)
{
  dbus_uint32_t *i;

  CHECK_SIGNATURE(msg, err, "u");

  i = malloc(sizeof(dbus_uint32_t));
  /* Actually emit the unsigned integer */
  dbus_message_get_args(msg, err,
                        DBUS_TYPE_UINT32, i,
                        DBUS_TYPE_INVALID);

  return i;
}

/**
 * @internal
 * @brief Callback for methods that return DBUS_TYPE_BOOLEAN
 */
void *
cb_nm_boolean(DBusMessage *msg, DBusError *err)
{
  dbus_bool_t *i;

  CHECK_SIGNATURE(msg, err, "b");

  i = malloc(sizeof(dbus_bool_t));
  /* Actually emit the unsigned integer */
  dbus_message_get_args(msg, err,
                        DBUS_TYPE_BOOLEAN, i,
                        DBUS_TYPE_INVALID);
  
  return i;
}

/**
 * @internal
 * @brief Callback for methods returning a single object path
 */
void *
cb_nm_object_path(DBusMessage *msg, DBusError *err)
{
  char *str;

  CHECK_SIGNATURE(msg, err, "o");

  /* Actually emit the object_path */
  dbus_message_get_args(msg, err,
                        DBUS_TYPE_OBJECT_PATH, &str,
                        DBUS_TYPE_INVALID);

  return str;
}
#endif


/**
 * @internal
 * @brief Callback for methods returning a list of object paths
 */
void *
cb_nm_object_path_list(DBusMessage *msg, DBusError *err)
{
  Eina_List *devices = NULL;
  char *dev;

  DBusMessageIter iter, sub;

  CHECK_SIGNATURE(msg, err, "ao");

  dbus_message_iter_init(msg, &iter);

  dbus_message_iter_recurse(&iter, &sub);
  while (dbus_message_iter_get_arg_type(&sub) != DBUS_TYPE_INVALID)
  {
    dev = NULL;
    if (!check_arg_type(&sub, 'o')) goto error;
    dbus_message_iter_get_basic(&sub, &dev);
    if (dev) devices = eina_list_append(devices, strdup(dev));
    dbus_message_iter_next(&sub);
  }

  printf("Hm: devices create = %p\n", devices);
  return devices;
error:
  EINA_LIST_FREE(devices, dev)
    free(dev);
  return NULL;
}

void
free_nm_object_path_list(void *data)
{
  Eina_List *list = data;
  char      *s;

  printf("Hm: devices delete = %p\n", list);
  EINA_LIST_FREE(list, s)
    free(s);
}

Eina_Hash *
parse_settings(DBusMessage *msg)
{
  Eina_Hash *settings;
  DBusMessageIter iter, a_iter;

  if (!dbus_message_has_signature(msg, "a{sa{sv}}")) return NULL;

  dbus_message_iter_init(msg, &iter);

  settings = eina_hash_string_small_new(EINA_FREE_CB(eina_hash_free));
  dbus_message_iter_recurse(&iter, &a_iter);
  while (dbus_message_iter_get_arg_type(&a_iter) != DBUS_TYPE_INVALID)
  {
    DBusMessageIter  d_iter, a2_iter;
    E_NM_Variant   *prop;
    const char      *name;
    Eina_Hash      *value;

    dbus_message_iter_recurse(&a_iter, &d_iter);
    if (!check_arg_type(&d_iter, 's')) goto error;
    dbus_message_iter_get_basic(&d_iter, &name);

    dbus_message_iter_next(&d_iter);
    if (!check_arg_type(&d_iter, 'a')) goto error;
    dbus_message_iter_recurse(&d_iter, &a2_iter);

    value = eina_hash_string_small_new(EINA_FREE_CB(property_free));
    eina_hash_add(settings, name, value);
    while (dbus_message_iter_get_arg_type(&a2_iter) != DBUS_TYPE_INVALID)
    {
      dbus_message_iter_recurse(&a2_iter, &d_iter);
      if (!check_arg_type(&d_iter, 's')) goto error;
      dbus_message_iter_get_basic(&d_iter, &name);
      dbus_message_iter_next(&d_iter);
      if (!check_arg_type(&d_iter, 'v')) goto error;
      prop = property_variant(&d_iter, NULL, NULL);
      if (prop) eina_hash_add(value, name, prop);
      dbus_message_iter_next(&a2_iter);
    }

    dbus_message_iter_next(&a_iter);
  }

  return settings;
error:
  eina_hash_free(settings);
  return NULL;
}

int
check_arg_type(DBusMessageIter *iter, char type)
{
  char sig;
 
  sig = dbus_message_iter_get_arg_type(iter);
  return sig == type;
}

void
property_data_free(Property_Data *data)
{
  if (data->object) free(data->object);
  free(data);
}

const char *
ip4_address2str(unsigned int address)
{
  static char buf[16];

  snprintf(buf, sizeof(buf), "%u.%u.%u.%u",
           ((address      ) & 0xff),
           ((address >> 8 ) & 0xff),
           ((address >> 16) & 0xff),
           ((address >> 24) & 0xff));
  return buf;
}

