/*
 * Copyright (C) 2009 Intel Corp.
 * Author: Forrest Zhao <forrest.zhao@intel.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <clutter/clutter.h>
#include "math.h"
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <errno.h>
#include <cairo/cairo.h>
#include <pango/pangocairo.h>
#include <GL/gl.h>
#include "engine.h"
#include "stroke.h"

#include "../clutter-gesture/gesture-events.h"

typedef enum
{
  ACTOR_STATE_IDLE = 0,
  ACTOR_STATE_IN_RECOGNIZE,
  ACTOR_STATE_IN_EMIT_EVENT
} actor_state_t;

struct engine_core
{
  GSList               *actors;
  recognize_callback_t cb;
  void                 *data;

  GSList               *stroke_list;
};

struct actor_core
{
  ClutterActor  *handle;
  guint         mask;

  actor_state_t state;
  GSList        *events;
  guint         num_events;
  guint         timer;

  recognize_callback_t cb;
  void                 *data;
};

static void release_cached_events(struct actor_core *actor_core)
{
  GSList *l;

  for (l = actor_core->events; l; l = l->next)
    {
      ClutterEvent *event;

      event = l->data;
      clutter_event_free(event);
    }

  g_slist_free(actor_core->events);
  actor_core->num_events = 0;
  actor_core->events = NULL;
}

static gboolean gesture_recognize_timeout(struct actor_core *actor_core)
{
  actor_core->cb(actor_core->handle, CLUTTER_EVENT, actor_core->events,
                             actor_core->num_events, actor_core->data);

  release_cached_events(actor_core);
  actor_core->state = ACTOR_STATE_IDLE;

  return FALSE;
}

ClutterGestureSlideEvent *new_gesture_slide_event(struct actor_core *actor_core,
                            ClutterGestureSlideDirection direction, guint speed)
{
  ClutterGestureSlideEvent *slide_event;
  ClutterEvent *event;

  slide_event = g_new0(ClutterGestureSlideEvent, 1);

  slide_event->type = GESTURE_SLIDE;

  event = g_slist_nth_data(actor_core->events, actor_core->num_events - 1);
  clutter_event_get_coords(event, &slide_event->x_end, &slide_event->y_end);
  slide_event->time = clutter_event_get_time(event);
  slide_event->stage = clutter_event_get_stage(event);
  /* TODO: ref/unref actor_core->handle */
  slide_event->source = actor_core->handle;

  event = g_slist_nth_data(actor_core->events, 0);
  clutter_event_get_coords(event, &slide_event->x_start, &slide_event->y_start);
  slide_event->start_time = clutter_event_get_time(event);

  slide_event->direction = direction;

  return slide_event;
}

static guint calculate_speed(struct actor_core *actor_core,
                            ClutterGestureSlideDirection direction)
{
  gfloat x, y, x1, y1;
  guint32 time, time1;
  ClutterEvent *event;
  double distance, speed;

  event = g_slist_nth_data(actor_core->events, 0);
  clutter_event_get_coords(event, &x, &y);
  time = clutter_event_get_time(event);

  event = g_slist_nth_data(actor_core->events, actor_core->num_events - 1);
  clutter_event_get_coords(event, &x1, &y1);
  time1 = clutter_event_get_time(event);

  if (direction == SLIDE_UP || direction == SLIDE_DOWN)
    {
      distance = y1 - y;
    }
  else
    {
      distance = x1 - x;
    }
  speed = distance/(time1 - time);
  speed = speed * 10;
  if (speed < 0)
    {
      speed = -speed;
    }

  if (speed > 10)
    speed = 10;
  g_print("y1 is %f, y is %f, time1 is %u, time is %u, the speed is %f\n",
         y1, y, time1, time, speed);

  return speed;
}

/* send out GESTURE_EVENT if a gesture is recognize,
 * otherwise replay the cached CLUTTER_EVENT */
static void gesture_recognize(struct engine_core *engine,
                              struct actor_core *actor_core)
{
  GSList *l;
  stroke_t *s = stroke_alloc(actor_core->num_events);
  gfloat x, y;
  double cost, score;
  guint direction = 0;
  void *gesture_event = NULL;
  GSList *gesture_event_list = NULL;

  /* TODO: recognize the gesture */
  for (l = actor_core->events; l; l = l->next)
    {
      ClutterEvent *event;

      event = l->data;
      clutter_event_get_coords(event, &x, &y);
      stroke_add_point(s, x, y);
      g_print("x/y is %f, %f, event type is %d\n", x, y,
              clutter_event_type(event));
    }
  stroke_finish(s);

  for (l = engine->stroke_list; l; l = l->next)
    {
      stroke_t *target_s = l->data;

      direction++;
      cost = stroke_compare(s, target_s, NULL, NULL);
      g_print("in gesture_recognize, cost is %f\n", cost);
      if (cost >= stroke_infinity)
        {
          continue;
        }
      score = MAX(1.0 - 2.5*cost, 0.0);
      g_print("in gesture_recognize, score is %f\n", score);
      if (score > 0.7)
        {
          guint speed;

          g_print("this is a match!!, direction is %d\n", direction);
          /* calculate speed */
          speed = calculate_speed(actor_core, direction);
          g_print("the speed is %d\n", speed);
          /* send out GESTURE_EVENT */
          gesture_event = new_gesture_slide_event(actor_core, direction, speed);
          gesture_event_list = g_slist_append(gesture_event_list, gesture_event);
          actor_core->cb(actor_core->handle, GESTURE_EVENT, gesture_event_list,
                         1, actor_core->data);
          goto done;
        }
    }

  actor_core->cb(actor_core->handle, CLUTTER_EVENT, actor_core->events,
                             actor_core->num_events, actor_core->data);

done:
  if (gesture_event_list)
  {
    g_free(gesture_event);
    g_slist_free(gesture_event_list);
  }
  stroke_free(s);
  release_cached_events(actor_core);
  actor_core->state = ACTOR_STATE_IDLE;
}

struct actor_core *find_actor(GSList *list, ClutterActor *actor)
{
  GSList *l;

  for (l = list; l; l = l->next)
    {
      struct actor_core *actor_core = l->data;

      if (actor_core->handle == actor)
        return actor_core;
    }

  return NULL;
}

struct coord
{
  gfloat x;
  gfloat y;
};

static void stroke_init(struct engine_core *engine)
{
  struct coord slide_down_coords[5] =
                {{454, 344}, {454, 345}, {453, 355}, {451, 362}, {451, 362}};
  struct coord slide_up_coords[5] =
                {{545, 443}, {545, 442}, {545, 426}, {545, 398}, {545, 388}};
  struct coord slide_left_coords[5] =
                {{376, 383}, {374, 383}, {358, 384}, {341, 384}, {332, 381}};
  struct coord slide_right_coords[5] =
                {{310, 389}, {312, 389}, {362, 389}, {411, 389}, {411, 389}};
  stroke_t *slidedown_s = stroke_alloc(5), *slideup_s = stroke_alloc(5),
           *slideright_s = stroke_alloc(5), *slideleft_s = stroke_alloc(5);
  gint i;

  for (i = 0; i < 5; i++)
    {
      stroke_add_point(slideup_s, slide_up_coords[i].x, slide_up_coords[i].y);
    }
  stroke_finish(slideup_s);
  engine->stroke_list = g_slist_append(engine->stroke_list, slideup_s);

  for (i = 0; i < 5; i++)
    {
      stroke_add_point(slidedown_s, slide_down_coords[i].x,
                                    slide_down_coords[i].y);
    }
  stroke_finish(slidedown_s);
  engine->stroke_list = g_slist_append(engine->stroke_list, slidedown_s);

  for (i = 0; i < 5; i++)
    {
      stroke_add_point(slideleft_s, slide_left_coords[i].x,
                                    slide_left_coords[i].y);
    }
  stroke_finish(slideleft_s);
  engine->stroke_list = g_slist_append(engine->stroke_list, slideleft_s);

  for (i = 0; i < 5; i++)
    {
      stroke_add_point(slideright_s, slide_right_coords[i].x,
                                     slide_right_coords[i].y);
    }
  stroke_finish(slideright_s);
  engine->stroke_list = g_slist_append(engine->stroke_list, slideright_s);
}

handle_t engine_init(recognize_callback_t cb, void *data)
{
  struct engine_core *engine;

  g_print("engine init is called\n");

  engine = g_new0(struct engine_core, 1);
  engine->cb = cb;
  engine->data = data;

  stroke_init(engine);

  return engine;
}

void engine_shutdown(handle_t handle)
{
  struct engine_core *engine = handle;
  GSList *l;

  for (l = engine->stroke_list; l; l = l->next)
    {
      stroke_t *s = l->data;
      stroke_free(s);
    }
  g_slist_free(engine->stroke_list);

  /* release actor_cores */
  for (l = engine->actors; l; l = l->next)
    {
      struct actor_core *actor_core;

      actor_core = l->data;
      g_object_unref(actor_core->handle);
      release_cached_events(actor_core);
      if (actor_core->timer)
        {
          g_source_remove(actor_core->timer);
          actor_core->timer = 0;
        }
    }
  g_slist_free(engine->actors);

  g_free(engine);

  return;
}

/* Return value: %0 if success, less than 0 if fail */
gint set_gesture_mask(handle_t handle, ClutterActor *target_actor, guint mask)
{
  struct engine_core *engine = handle;
  struct actor_core *actor_core;

  if (!mask)
    {
      actor_core = find_actor(engine->actors, target_actor);
      if (!actor_core)
        return -ENOENT;
      engine->actors = g_slist_remove(engine->actors, actor_core);
      g_object_unref(target_actor);
      /* release events, stop timer */
      release_cached_events(actor_core);

      if (actor_core->timer)
        {
          g_source_remove(actor_core->timer);
          actor_core->timer = 0;
        }

      g_free(actor_core);
      return 0;
    }

  actor_core = find_actor(engine->actors, target_actor);
  if (!actor_core)
    {
      g_object_ref(target_actor);
      actor_core = g_new0(struct actor_core, 1);
      actor_core->handle = target_actor;
      actor_core->mask = mask;
      actor_core->state = ACTOR_STATE_IDLE;
      actor_core->num_events = 0;
      actor_core->timer = 0;
      actor_core->cb = engine->cb;
      actor_core->data = engine->data;
      engine->actors = g_slist_append(engine->actors, actor_core);
    }
  else
    {
      actor_core->mask = mask;
    }

  return 0;
}

static gboolean state_idle_handle_event(struct engine_core *engine,
                struct actor_core *actor_core, ClutterEvent *event)
{
  ClutterButtonEvent *buttev;

  if (clutter_event_type(event) != CLUTTER_BUTTON_PRESS)
    return FALSE;

  buttev = (ClutterButtonEvent *)event;

  /* "single left click" triggers the gesture recognition */
  if (clutter_event_get_button(event) != 1 || buttev->click_count != 1)
    return FALSE;

  /* start the timer */
  actor_core->timer = g_timeout_add(300,
                      (GSourceFunc) gesture_recognize_timeout, actor_core);
  
  actor_core->events = g_slist_append(actor_core->events,
                                      clutter_event_copy(event));
  actor_core->num_events++;
  
  actor_core->state = ACTOR_STATE_IN_RECOGNIZE;

  return TRUE;
}

static gboolean state_in_recognize_handle_event(struct engine_core *engine,
                struct actor_core *actor_core, ClutterEvent *event)
{
  ClutterEvent *start_event;
  gfloat start_x, start_y, end_x, end_y;

  if (clutter_event_type(event) == CLUTTER_MOTION)
    {
      actor_core->events = g_slist_append(actor_core->events,
                                          clutter_event_copy(event));
      actor_core->num_events++;
      return TRUE;
    }

  if (clutter_event_type(event) != CLUTTER_BUTTON_RELEASE)
    {
      return FALSE;
    }

  /* add the last CLUTTER_BUTTON_RELEASE event to event queue */
  actor_core->events = g_slist_append(actor_core->events,
                                          clutter_event_copy(event));
  actor_core->num_events++;

   /* stop the timer and start recognition */
  if (actor_core->timer)
    {
      g_source_remove(actor_core->timer);
      actor_core->timer = 0;
    }

  start_event = g_slist_nth_data(actor_core->events, 0);
  clutter_event_get_coords(start_event, &start_x, &start_y);
  clutter_event_get_coords(event, &end_x, &end_y);
  if (((start_x == end_x) && (start_y == end_y)) ||
      (actor_core->num_events <= 3))
    {
        actor_core->cb(actor_core->handle, CLUTTER_EVENT, actor_core->events,
                             actor_core->num_events, actor_core->data);

        release_cached_events(actor_core);
        actor_core->state = ACTOR_STATE_IDLE;
    }
  else
    {
      gesture_recognize(engine, actor_core);
    }

  return TRUE;
}

/* Return value: %TRUE if event is accepted by engine; %FALSE otherwise */
gboolean fill_event(handle_t handle, ClutterActor *target_actor,
			ClutterEvent *event)
{
  struct engine_core *engine = handle;
  struct actor_core *actor_core;
  gboolean ret = FALSE;

  actor_core = find_actor(engine->actors, target_actor);
  if (!actor_core)
    {
      g_print("in fill_event(), actor_core not found\n");
      return FALSE;
    }

  switch (actor_core->state)
    {
    case ACTOR_STATE_IDLE:
      ret = state_idle_handle_event(engine, actor_core, event);
      break;
    case ACTOR_STATE_IN_RECOGNIZE:
      ret = state_in_recognize_handle_event(engine, actor_core, event);
      break;
    default:
      break;
    }

  g_print("in fill_event(), ret is %d\n", ret);
  return ret;
}
