/*
 * tcpconnect.cpp
 *
 *  Created on: 27.02.2010
 *      Author: ed
 */

#define LOCAL_DEBUG
#include "debug.h"

#include "meta.h"
#include "tcpconnect.h"

#include "acfg.h"
#include "caddrinfo.h"
#include <signal.h>
#include "sockio.h"

using namespace MYSTD;

tcpconnect::tcpconnect() :	m_conFd(-1)
{
	m_proxy = acfg::proxy_info.sHost.empty() ? NULL : &acfg::proxy_info;
}

tcpconnect::~tcpconnect()
{
	ldbg("terminating outgoing connection class");
	_Disconnect();
}



static int connect_timeout(int sockfd, const struct sockaddr *addr, socklen_t addrlen, time_t timeout)
{
	long stflags;
	struct timeval tv;
	fd_set wfds;
	int res;

	tv.tv_sec = timeout;
	tv.tv_usec = 0;

	if ((stflags = fcntl(sockfd, F_GETFL, NULL)) < 0)
		return -1;

	// Set to non-blocking mode.
	if (fcntl(sockfd, F_SETFL, stflags|O_NONBLOCK) < 0)
		return -1;

	res = connect(sockfd, addr, addrlen);
	if (res < 0) {
		if (EINPROGRESS == errno) {
			for (;;) {
				// Wait for connection.
				FD_ZERO(&wfds);
				FD_SET(sockfd, &wfds);
				res = select(sockfd+1, NULL, &wfds, NULL, &tv);
				if (res < 0) {
					if (EINTR != errno)
						return -1;
				} else if (res > 0) {
					// Socket selected for writing.
					int err;
					socklen_t optlen = sizeof(err);

					if (getsockopt(sockfd, SOL_SOCKET, SO_ERROR, (void *)&err, &optlen) < 0)
						return -1;

					if (err) {
						errno = err;
						return -1;
					}

					break;
				} else {
					// Timeout.
					errno = ETIMEDOUT;
					return -1;
				}
			}
		} else {
			return -1;
		}
	}

	// Set back to original mode, which may or may not have been blocking.
	if (fcntl(sockfd, F_SETFL, stflags) < 0)
		return -1;

	return 0;
}


bool tcpconnect::_Connect(const string & sHostname, string & sErrorMsg)
{
	_Disconnect();
	ldbg("Resolving " + sHostname);
	CAddrInfo::SPtr dns = m_proxy ?
			CAddrInfo::CachedResolve(m_proxy->sHost, m_proxy->sPort, sErrorMsg)
			: CAddrInfo::CachedResolve(sHostname, acfg::remoteport, sErrorMsg);
	if(!dns)
	{
		return false; // sErrorMsg got the info already, no other chance to fix it
		ldbg("DNS error");
	}

	::signal(SIGPIPE, SIG_IGN);


	for (struct addrinfo *pInfo=dns->m_bestInfo; pInfo; pInfo = pInfo->ai_next)
	{
		ldbg("Creating socket for " << sHostname);

		if (pInfo->ai_socktype != SOCK_STREAM || pInfo->ai_protocol != IPPROTO_TCP)
			continue;

		m_conFd = ::socket(pInfo->ai_family, pInfo->ai_socktype, pInfo->ai_protocol);
		if(m_conFd<0)
			continue;

#ifndef NO_TCP_TUNNING
		{
			int yes(1);
			::setsockopt(m_conFd,SOL_SOCKET,SO_REUSEADDR,&yes,sizeof(yes));
			::setsockopt(m_conFd, IPPROTO_TCP, TCP_NODELAY, &yes, sizeof(yes));
		}
#endif

		if(::connect_timeout(m_conFd, pInfo->ai_addr, pInfo->ai_addrlen, acfg::nettimeout) < 0)
			continue;

		ldbg("connect() ok");
		set_nb(m_conFd);
		return true;
	}

	// format the last available error message for the user
	aclog::errnoFmter fer;
	sErrorMsg="500 Connection failure: ";
	sErrorMsg+=fer.msg;
	ldbg("Force reconnect, con. failure");
	_Disconnect();
	return false;
}

void tcpconnect::_Disconnect()
{
	dbgline;
	if(m_conFd<0)
		return;
	::shutdown(m_conFd, O_RDWR);
	forceclose(m_conFd);
	m_conFd=-1;
}
