/* -*- Mode: javascript; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* ***** BEGIN LICENSE BLOCK *****
 *	 Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 * 
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is android-account-sync-extension.
 *
 * The Initial Developer of the Original Code is
 * Canonical, Ltd.
 * Portions created by the Initial Developer are Copyright (C) 2012
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 * Renato Araujo Oliveira Filho <renato@canonical.com>
 * Ricardo Mendoza <ricmm@canonical.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 * 
 * ***** END LICENSE BLOCK ***** */

const Cu = Components.utils;
const Ci = Components.interfaces;

Cu.import("resource://gre/modules/ctypes.jsm");
Cu.import("resource://gre/modules/Services.jsm");

Cu.import("resource://androidaccountsync/LibGLib.jsm");
Cu.import("resource://androidaccountsync/LibAndroidAccountManagerProxy.jsm");

var EXPORTED_SYMBOLS = [ "AndroidAccountSync" ];

var createThunderBirdAccount = function AAS_createThunderBirdAccount(accountData) {
  let account = ctypes.cast(accountData, LibAccountManagerProxy.AccountInfo.ptr);
  var accountManager = Components.classes["@mozilla.org/messenger/account-manager;1"]
                        .getService(Ci.nsIMsgAccountManager);
  
  var identity = accountManager.createIdentity();

  var vFullName = LibAccountManagerProxy.getAccountInfoField(account, LibAccountManagerProxy.ACCOUNT_SENDER_NAME);
  var vEmail = LibAccountManagerProxy.getAccountInfoField(account, LibAccountManagerProxy.ACCOUNT_EMAIL);
  var vLogin = LibAccountManagerProxy.getAccountInfoField(account, LibAccountManagerProxy.ACCOUNT_INCOMING_HOST_LOGIN);
  var vHost = LibAccountManagerProxy.getAccountInfoField(account, LibAccountManagerProxy.ACCOUNT_INCOMING_HOST_ADDRESS);
  var vHostType = LibAccountManagerProxy.getAccountInfoField(account, LibAccountManagerProxy.ACCOUNT_INCOMING_HOST_PROTOCOL);
  var vHostPort = LibAccountManagerProxy.getAccountInfoField(account, LibAccountManagerProxy.ACCOUNT_INCOMING_HOST_PORT);

  identity.email = LibGLib.g_variant_get_string(vEmail).readString();
  identity.fullName = LibGLib.g_variant_get_string(vFullName).readString();

  var inServer = accountManager.createIncomingServer(LibGLib.g_variant_get_string(vLogin).readString(),
                                                   LibGLib.g_variant_get_string(vHost).readString(),
                                                   LibGLib.g_variant_get_string(vHostType).readString());
  
  inServer.port = LibGLib.g_variant_get_int32(vHostPort);

  if (inServer.port == 993 || inServer.port == 995) {
    inServer.socketType = 3;
    inServer.isSecure = true;
  } else {
    inServer.socketType = 1;
  }
  
  vHost = LibAccountManagerProxy.getAccountInfoField(account, LibAccountManagerProxy.ACCOUNT_OUTGOING_HOST_ADDRESS);
  vHostType = LibAccountManagerProxy.getAccountInfoField(account, LibAccountManagerProxy.ACCOUNT_OUTGOING_HOST_PROTOCOL);
  vHostPort = LibAccountManagerProxy.getAccountInfoField(account, LibAccountManagerProxy.ACCOUNT_OUTGOING_HOST_PORT);
  vLogin = LibAccountManagerProxy.getAccountInfoField(account, LibAccountManagerProxy.ACCOUNT_OUTGOING_HOST_LOGIN);

  var smtpService = Components.classes["@mozilla.org/messengercompose/smtp;1"]
                        .getService(Components.interfaces.nsISmtpService);

  var outServer = smtpService.createSmtpServer();
  outServer.hostname = LibGLib.g_variant_get_string(vHost).readString();
  outServer.port = LibGLib.g_variant_get_int32(vHostPort);
  outServer.username = LibGLib.g_variant_get_string(vLogin).readString();
  identity.smtpServerKey = outServer.key;

  var account = accountManager.createAccount();
  account.incomingServer = inServer;
  identity.setCharAttribute("remote", "true");
  account.addIdentity(identity);
  account.defaultIdentity = identity;

  dump("Thunderbird account created: " + identity.email + "\n");
}

var updateThunderBirdAccount = function AAS_updateThunderBirdAccount(accountData, updateMask) {
  var accountManager = Components.classes["@mozilla.org/messenger/account-manager;1"]
                        .getService(Ci.nsIMsgAccountManager);
  let account = ctypes.cast(accountData, LibAccountManagerProxy.AccountInfo.ptr);
  var accounts = accountManager.accounts;

  var vEmail = LibAccountManagerProxy.getAccountInfoField(account, LibAccountManagerProxy.ACCOUNT_EMAIL);
  
  for (var i = 0; i < accounts.Count(); i++) {
    var local = accounts.QueryElementAt(i, Components.interfaces.nsIMsgAccount);
    server = local.incomingServer;
   
    if (server.username == "nobody")
      continue;

    if (server.username == LibGLib.g_variant_get_string(vEmail).readString())
      break;
  }

  if (updateMask & (1 << LibAccountManagerProxy.FIELD_SENDER_NAME)) {
    var vSenderName = LibAccountManagerProxy.getAccountInfoField(account, LibAccountManagerProxy.ACCOUNT_SENDER_NAME);
    local.defaultIdentity.fullName = LibGLib.g_variant_get_string(vSenderName).readString();
  }
  if (updateMask & (1 << LibAccountManagerProxy.INC_HOST_LOGIN)) {
    var vHostLogin = LibAccountManagerProxy.getAccountInfoField(account, LibAccountManagerProxy.ACCOUNT_INCOMING_HOST_LOGIN);
    var inServer = accountManager.createIncomingServer(LibGLib.g_variant_get_string(vHostLogin).readString(),
                                                       local.incomingServer.hostName,
                                                       local.incomingServer.type);
    inServer.port = local.incomingServer.port;
    local.incomingServer = inServer;
  }
  if (updateMask & (1 << LibAccountManagerProxy.INC_HOST_PROTOCOL)) {
    var vHostType = LibAccountManagerProxy.getAccountInfoField(account, LibAccountManagerProxy.ACCOUNT_INCOMING_HOST_PROTOCOL);
    var inServer = accountManager.createIncomingServer(local.incomingServer.username,
                                                       local.incomingServer.hostName,
                                                       LibGLib.g_variant_get_string(vHostType).readString());
    inServer.port = local.incomingServer.port;
    local.incomingServer = inServer;
  }
  if (updateMask & (1 << LibAccountManagerProxy.INC_HOST_PORT)) {
    var vHostPort = LibAccountManagerProxy.getAccountInfoField(account, LibAccountManagerProxy.ACCOUNT_INCOMING_HOST_PORT);
    local.incomingServer.port = LibGLib.g_variant_get_int32(vHostPort);
  }
  if (updateMask & (1 << LibAccountManagerProxy.INC_HOST_ADDRESS)) {
    var vHost = LibAccountManagerProxy.getAccountInfoField(account, LibAccountManagerProxy.ACCOUNT_INCOMING_HOST_ADDRESS);
    var inServer = accountManager.createIncomingServer(local.incomingServer.username,
                                                       LibGLib.g_variant_get_string(vHost).readString(),
                                                       local.incomingServer.type);
    inServer.port = local.incomingServer.port;
    local.incomingServer = inServer;
  }

  var outServer = smtpService.getServerByKey(local.defaultIdentity.smtpServerKey);

  if (updateMask & (1 << LibAccountManagerProxy.OUT_HOST_LOGIN)) {
    var vHostLogin = LibAccountManagerProxy.getAccountInfoField(account, LibAccountManagerProxy.ACCOUNT_OUTGOING_HOST_LOGIN);
    outServer.hostname = LibGLib.g_variant_get_string(vHostLogin).readString();
  }
  if (updateMask & (1 << LibAccountManagerProxy.OUT_HOST_PORT)) {
    var vHostPort = LibAccountManagerProxy.getAccountInfoField(account, LibAccountManagerProxy.ACCOUNT_OUTGOING_HOST_PORT);
    outServer.port = LibGLib.g_variant_get_int32(vHostPort);
  }
  if (updateMask & (1 << LibAccountManagerProxy.OUT_HOST_ADDRESS)) {
    var vHost = LibAccountManagerProxy.getAccountInfoField(account, LibAccountManagerProxy.ACCOUNT_OUTGOING_HOST_ADDRESS);
    outServer.hostname = LibGLib.g_variant_get_string(vHost).readString();
  }
  dump("Account updated: " + LibGLib.g_variant_get_string(vEmail).readString() + "\n");
}

var removeThunderBirdAccount = function AAS_removeThunderBirdAccount(accountData) {
  var accountManager = Components.classes["@mozilla.org/messenger/account-manager;1"]
                        .getService(Ci.nsIMsgAccountManager);
  let account = ctypes.cast(accountData, LibAccountManagerProxy.AccountInfo.ptr);

  var vEmail = LibAccountManagerProxy.getAccountInfoField(account, LibAccountManagerProxy.ACCOUNT_EMAIL);
  var vHost = LibAccountManagerProxy.getAccountInfoField(account, LibAccountManagerProxy.ACCOUNT_INCOMING_HOST_ADDRESS);
  var vHostType = LibAccountManagerProxy.getAccountInfoField(account, LibAccountManagerProxy.ACCOUNT_INCOMING_HOST_PROTOCOL);

  var server = accountManager.FindServer(LibGLib.g_variant_get_string(vEmail).readString(),
                                         LibGLib.g_variant_get_string(vHost).readString(),
                                         LibGLib.g_variant_get_string(vHostType).readString());
  if (server == null) {
    dump("Server not found\n");
    return;
  }

  var account = accountManager.FindAccountForServer(server);
  if (account == null) {
    dump("Account not found\n");
    return;
  }

  identity = account.defaultIdentity;

  if (!identity.getCharAttribute("remote"))
    return 0;

  accountManager.removeAccount(account);
  dump("Thunderbird account removed:" + email + "\n");
}



var AndroidAccountSync = {

  _initialized: false,
  _proxy: null,
  _accountCreatedCallback: null,
  _accountRemovedCallback: null,

  _serverToAccount: function AAS_serverToAccount (server, account, local) {   
    var vEmail = LibGLib.g_variant_new_string(local.defaultIdentity.email);
    var vLogin = LibGLib.g_variant_new_string(server.realUsername);
    var vHostName = LibGLib.g_variant_new_string(server.realHostName);
    var vHostType = LibGLib.g_variant_new_string(server.type);
    var vHostPort = LibGLib.g_variant_new_int32(server.port);
    var vFullName = LibGLib.g_variant_new_string(local.defaultIdentity.fullName);

    LibAccountManagerProxy.setAccountInfoField(account, LibAccountManagerProxy.ACCOUNT_EMAIL, vEmail);
    LibAccountManagerProxy.setAccountInfoField(account, LibAccountManagerProxy.ACCOUNT_SENDER_NAME, vFullName);
    LibAccountManagerProxy.setAccountInfoField(account, LibAccountManagerProxy.ACCOUNT_INCOMING_HOST_ADDRESS, vHostName);
    LibAccountManagerProxy.setAccountInfoField(account, LibAccountManagerProxy.ACCOUNT_INCOMING_HOST_LOGIN, vLogin);
    LibAccountManagerProxy.setAccountInfoField(account, LibAccountManagerProxy.ACCOUNT_INCOMING_HOST_PROTOCOL, vHostType);
    LibAccountManagerProxy.setAccountInfoField(account, LibAccountManagerProxy.ACCOUNT_INCOMING_HOST_PORT, vHostPort);

    var smtpService = Components.classes["@mozilla.org/messengercompose/smtp;1"]
                        .getService(Components.interfaces.nsISmtpService);

    var outServer = smtpService.getServerByKey(local.defaultIdentity.smtpServerKey);

    vHostName = LibGLib.g_variant_new_string(outServer.hostname);
    vHostPort = LibGLib.g_variant_new_int32(outServer.port);
    vLogin = LibGLib.g_variant_new_string(outServer.username);

    LibAccountManagerProxy.setAccountInfoField(account, LibAccountManagerProxy.ACCOUNT_OUTGOING_HOST_ADDRESS, vHostName);
    LibAccountManagerProxy.setAccountInfoField(account, LibAccountManagerProxy.ACCOUNT_OUTGOING_HOST_LOGIN, vLogin);
    LibAccountManagerProxy.setAccountInfoField(account, LibAccountManagerProxy.ACCOUNT_OUTGOING_HOST_PORT, vHostPort);
    
    LibGLib.g_variant_unref(vEmail);
    LibGLib.g_variant_unref(vHostName);
    LibGLib.g_variant_unref(vHostType);
    LibGLib.g_variant_unref(vLogin);
    LibGLib.g_variant_unref(vHostPort);
    LibGLib.g_variant_unref(vFullName);
  },

  _onCreatedAccount: function AAS_accountCreated (proxy, email, uData) {
    let emailStr = email.readString();
    let proxyPtr = ctypes.cast(proxy, LibAccountManagerProxy.AndroidAccountManagerProxy.ptr);
    let account = LibAccountManagerProxy.getAccount(proxyPtr, emailStr);
    if (!account.isNull()) {
      // Create account on thunderbird
      createThunderBirdAccount(account); 

      // Register account locally
      LibAccountManagerProxy.registerAccount(proxyPtr, account);
    } else {
      dump("Account created ERROR:\n");
    }
  },

  _onAccountUpdated: function AAS_updateThunderBirdAccount(proxy, email, updateMask) {
    let emailStr = email.readString();
    let proxyPtr = ctypes.cast(proxy, LibAccountManagerProxy.AndroidAccountManagerProxy.ptr);
    let account = LibAccountManagerProxy.getAccount(proxyPtr, emailStr);

    if (!account.isNull())
        updateThunderBirdAccount(account, updateMask);
  },
    
  _onRemovedAccount: function AAS_accountRemoved (proxy, email, uData) {
    let proxyPtr = ctypes.cast(proxy, LibAccountManagerProxy.AndroidAccountManagerProxy.ptr);
    let account = LibAccountManagerProxy.getAccount(proxyPtr, email.readString());

    if (!account.isNull()) {
      if (!removeThunderBirdAccount(account))
        return;
      
      var vEmail = LibAccountManagerProxy.getAccountInfoField(account, LibAccountManagerProxy.ACCOUNT_EMAIL);

      LibAccountManagerProxy.unregisterAccount(proxyPtr,
                                               LibGLib.g_variant_get_string(vEmail).readString());
    }
  },

  syncAndroidAccounts: function () {
    var accounts = this._accountManager.accounts;
    
    for (var i = 0; i < accounts.Count(); i++) {
        var local = accounts.QueryElementAt(i, Components.interfaces.nsIMsgAccount);
        server = local.incomingServer;

        if (server.username == "nobody")
            continue;
       
	identity = local.defaultIdentity; 
	if (!identity.getCharAttribute("remote"))
	    return 0;

        var localAcc = LibAccountManagerProxy.newAccountInfo();
        this._serverToAccount(server, localAcc, local);
        var serverAcc = LibAccountManagerProxy.getAccount(this._proxy, server.username);

        LibAccountManagerProxy.unregisterAccount(this._proxy, server.username);
        LibAccountManagerProxy.registerAccount(this._proxy, localAcc);
        LibAccountManagerProxy.handleUpdate(this._proxy, serverAcc, localAcc);
    }
  },

  onServerLoaded: function (server) {
    // skip unsupported protocol types
    if ((server.type == "pop3") || (server.type == "imap")) {
      // Only accept gmail accounts (TODO: fix that on Android side)
      if (server.hostName == "pop.gmail.com") {

        let accountData = LibAccountManagerProxy.getAccount(this._proxy, server.username);

        // Account does not exists on android side
        if (accountData.isNull()) {
          let account = LibAccountManagerProxy.newAccountInfo();
          this._serverToAccount(server, account);
          LibAccountManagerProxy.createAccount(this._proxy, account);
          dump("Account created on android: " + server.username + "\n");
        }
      }
    }
  },

  onServerUnloaded: function (server) {
    if (!this._proxy.isNull()) {
      LibAccountManagerProxy.unregisterAccount(this._proxy, server.username);
    }
  },

  onServerChanged: function (server) {
    let account = LibAccountManagerProxy.getAccount(this._proxy, server.username);

    if (!account.isNull()) {
      this._serverToAccount(server, account);
      LibAccountManagerProxy.updateAccount(this._proxy, account);
      dump("Account Updated\n");
    } else {
      dump("Account not found: " + server.username + "\n");
    }
  },

  observe: function(aSubject, aTopic, aData) {
    if (aTopic != "quit-application-granted") {
      if (aSubject.document.getElementById("accountManager"))
        this.syncAndroidAccounts();
      return;
    }
    this.shutdown();
  },

  shutdown: function PROXY_shutdown() {
    this._accountManager.removeIncomingServerListener(this);
    dump("Account sync: shutdown\n");
  },

  init: function AndroidAccountSync_init() {
    dump("AndroidAccountSync: Start\n");
    if (this._initialized)
      return;

    this._accountManager = Components.classes["@mozilla.org/messenger/account-manager;1"].getService(Ci.nsIMsgAccountManager)
    this._proxy = LibAccountManagerProxy.newAccountManagerProxy();
    
    // Register current accounts
    var accounts = this._accountManager.accounts;
    
    for (var i = 0; i < accounts.Count(); i++) {
        var local = accounts.QueryElementAt(i, Components.interfaces.nsIMsgAccount);
        server = local.incomingServer;

        if (server.username == "nobody")
            continue;

        dump("Register account: " + server.username + "\n");
        var account = LibAccountManagerProxy.newAccountInfo();
        this._serverToAccount(server, account, local);
        LibAccountManagerProxy.registerAccount(this._proxy, account);
        dump("Account registered: " + server.username + "\n");
    }
    
    // connect signal account-created
    let accountCreatedCB = ctypes.FunctionType(ctypes.default_abi,
                                              ctypes.void_t,
                                              [LibGLib.gpointer,
                                               LibGLib.gchar.ptr,
                                               LibGLib.gpointer]).ptr;    
    this._accountCreatedCallback = accountCreatedCB(this._onCreatedAccount);      
    LibGLib.g_signal_connect(this._proxy, "account-created", this._accountCreatedCallback, null);

    // connect signal account-updated
    let accountUpdatedCB = ctypes.FunctionType(ctypes.default_abi,
                                               ctypes.void_t,
                                               [LibGLib.gpointer,
                                                LibGLib.gchar.ptr,
                                                ctypes.uint32_t]).ptr;
    this._accountUpdatedCallback = accountUpdatedCB(this._onAccountUpdated);
    LibGLib.g_signal_connect(this._proxy, "account-updated", this._accountUpdatedCallback, null);

    // connect signal account-removed
    let accountRemovedCB = ctypes.FunctionType(ctypes.default_abi,
                                              ctypes.void_t,
                                              [LibGLib.gpointer,
                                               LibGLib.gchar.ptr,
                                               LibGLib.gpointer]).ptr;
    this._accountRemovedCallback = accountRemovedCB(this._onRemovedAccount);
    LibGLib.g_signal_connect(this._proxy, "account-removed", this._accountRemovedCallback, null);

    LibAccountManagerProxy.startSync(this._proxy);
 
    // register server change listener
    this._accountManager.addIncomingServerListener(this);

    // Establish shutdown hook
    Services.obs.addObserver(this, "quit-application-granted", false);
    Services.obs.addObserver(this, "domwindowclosed", false);

    this._initialized = true;
    dump("AndroidAccountSync: INITIALIZED\n");
  },
}
